<?php
/*
 * Plugin Name: Screenfeed Updater
 * Plugin URI: https://www.screenfeed.fr
 * Description: Updater for plugins provided directly from the screenfeed's server.
 * Plugin version: 1.2.2
 * Author: Grégory Viguier
 * Author URI: https://www.screenfeed.fr/greg/
 * License: GPLv3
 * License URI: https://www.screenfeed.fr/gpl-v3.txt
 *
 * The filter 'sf_plugins_updates' waits for an array like this:
 * array(
 *     '/absolute/path/to/plugin.php' => 'plugin.version',
 *     '/absolute/path/to/other_plugin.php' => 'other.plugin.version',
 * )
 */

if ( ! defined( 'ABSPATH' ) ) {
	die( 'Cheatin\' uh?' );
}

/*------------------------------------------------------------------------------------------------*/
/* !EXCLUDE SF PLUGINS FROM WP UPDATES. ========================================================= */
/*------------------------------------------------------------------------------------------------*/

// Requests sent to WP servers

if ( ! function_exists( 'sf_updater_exclude_from_repo' ) ) :
add_filter( 'http_request_args', 'sf_updater_exclude_from_repo', 5, 2 );

function sf_updater_exclude_from_repo( $r, $url ) {
	// Not a plugin update request.
	if ( 0 !== strpos( $url, 'http://api.wordpress.org/plugins/update-check/' ) && 0 !== strpos( $url, 'https://api.wordpress.org/plugins/update-check/' ) ) {
		return $r;
	}

	$sf_plugins = apply_filters( 'sf_plugins_updates', array() );

	// No SF plugins to remove or error.
	if ( empty( $sf_plugins ) || ! is_array( $sf_plugins ) || empty( $r['body']['plugins'] ) ) {
		return $r;
	}

	$is_wp37 = ! is_serialized( $r['body']['plugins'] );

	if ( ! $is_wp37 ) {
		// WP < 3.7 (http)
		$plugins = unserialize( $r['body']['plugins'] );
	}
	else {
		// WP 3.7+ (https)
		$plugins = json_decode( $r['body']['plugins'] );
	}

	foreach ( $sf_plugins as $sf_plugin => $version ) {

		$sf_plugin = plugin_basename( $sf_plugin );

		if ( is_array( $plugins->plugins ) ) {
			unset( $plugins->plugins[ $sf_plugin ] );
		}
		else {
			unset( $plugins->plugins->$sf_plugin );
		}

		if ( is_object( $plugins->active ) ) {
			$plugins->active = array_values( (array) $plugins->active );

			if ( false !== ( $sf_plugin_pos = array_search( $sf_plugin, $plugins->active ) ) ) {
				unset( $plugins->active[ $sf_plugin_pos ] );
			}

			$plugins->active = (object) $plugins->active;
		}
		elseif ( is_array( $plugins->active ) && false !== ( $sf_plugin_pos = array_search( $sf_plugin, $plugins->active ) ) ) {
			unset( $plugins->active[ $sf_plugin_pos ] );
			$plugins->active = array_values( $plugins->active );
		}
	}

	if ( ! $is_wp37 ) {
		$r['body']['plugins'] = serialize( $plugins );
	}
	else {
		$r['body']['plugins'] = json_encode( $plugins );
	}

	return $r;
}
endif;


/*------------------------------------------------------------------------------------------------*/
/* !CHECK SF UPDATES TWICE A DAY, OR EVERY TIME WE VISIT THE UPDATES PAGES. ===================== */
/*------------------------------------------------------------------------------------------------*/

if ( ! function_exists( 'sf_updater_check_update' ) ) :
add_action( 'plugins_loaded', 'sf_updater_check_update' );

function sf_updater_check_update() {
	global $pagenow;

	if ( 'plugins.php' === $pagenow || 'update.php' === $pagenow || 'update-core.php' === $pagenow ) {
		add_filter( 'site_transient_update_plugins', 'sf_updater_site_transient_update_plugins' );
	}
	else {
		$current = get_site_transient( 'update_sf' );
		$timeout = apply_filters( 'sf_updater_check_update', 12 * HOUR_IN_SECONDS );

		if ( ! empty( $current ) && $timeout > ( time() - $current ) ) {
			return;
		}

		add_filter( 'site_transient_update_plugins', 'sf_updater_site_transient_update_plugins' );
	}
}
endif;


if ( ! function_exists( 'sf_updater_site_transient_update_plugins' ) ) :
function sf_updater_site_transient_update_plugins( $current ) {
	static $updates = null;

	if ( ! is_object( $current ) ) {
		$current = new stdClass;
	}

	if ( empty( $current->response ) || ! is_array( $current->response ) ) {
		$current->response = array();
	}

	if ( is_null( $updates ) ) {

		$updates    = array();
		$sf_plugins = apply_filters( 'sf_plugins_updates', array() );

		if ( ! empty( $sf_plugins ) && is_array( $sf_plugins ) ) {

			$last_checked = ! empty( $current->last_checked ) ? $current->last_checked : time();

			set_site_transient( 'update_sf', $last_checked );

			foreach ( $sf_plugins as $sf_plugin => $sf_version ) {

				$plugin_basename = plugin_basename( $sf_plugin );
				$plugin_folder   = dirname( $plugin_basename );
				$request         = array( 'slug' => $plugin_folder, 'fields' => array( 'sections' => false ) );
				$response        = sf_updater_remote_plugin_infos( $request );

				if ( is_wp_error( $response ) ) {
					continue;
				}

				$plugin = sf_updater_remote_retrieve_body( $response );

				if ( empty( $plugin->download_link ) || empty( $plugin->version ) || version_compare( $plugin->version, $sf_version ) <= 0 ) {
					unset( $current->response[ $plugin_basename ] );
					continue;
				}

				$updates[ $plugin_basename ] = (object) array(
					'slug'        => $plugin_folder,
					'plugin'      => $plugin_basename,
					'new_version' => $plugin->version,
					'url'         => ( ! empty( $plugin->homepage ) ? esc_url( $plugin->homepage ) : '' ),
					'package'     => esc_url( $plugin->download_link ),
				);

			}

		}

	}

	if ( count( $updates ) ) {
		$current->response = array_merge( $current->response, $updates );
		set_site_transient( 'update_plugins', $current );
	}

	return $current;
}
endif;


/*------------------------------------------------------------------------------------------------*/
/* !REPLACE THE REQUEST TO THE WP SERVERS WITH ONE TO OUR. ====================================== */
/*------------------------------------------------------------------------------------------------*/

if ( ! function_exists( 'sf_updater_pull_info' ) ) :
add_filter( 'plugins_api', 'sf_updater_pull_info', 10, 3 );

function sf_updater_pull_info( $res, $action, $args ) {
	if ( 'plugin_information' !== $action ) {
		return $res;
	}

	$sf_plugins = apply_filters( 'sf_plugins_updates', array() );

	if ( ! count( $sf_plugins ) ) {
		return $res;
	}

	$sf_plugins = array_map( 'basename', array_map( 'dirname', array_keys( $sf_plugins ) ) );

	if ( in_array( $args->slug, $sf_plugins ) ) {
		return sf_updater_remote_retrieve_body( sf_updater_remote_plugin_infos( $args ) );
	}

	return $res;
}
endif;


/*------------------------------------------------------------------------------------------------*/
/* !ADD THE "VIEW DETAILS" LINK. ================================================================ */
/*------------------------------------------------------------------------------------------------*/

if ( ! function_exists( 'sf_updater_add_plugin_slug' ) ) :
add_filter( 'all_plugins', 'sf_updater_add_plugin_slug' );

function sf_updater_add_plugin_slug( $plugins_all ) {

	$sf_plugins = apply_filters( 'sf_plugins_updates', array() );

	if ( empty( $sf_plugins ) || ! is_array( $sf_plugins ) ) {
		return $plugins_all;
	}

	foreach ( $sf_plugins as $sf_plugin => $sf_version ) {

		$plugin_basename = plugin_basename( $sf_plugin );

		if ( isset( $plugins_all[ $plugin_basename ] ) ) {
			$plugins_all[ $plugin_basename ]['slug'] = dirname( $plugin_basename );
		}

	}

	return $plugins_all;
}
endif;


/*------------------------------------------------------------------------------------------------*/
/* !UTILITIES =================================================================================== */
/*------------------------------------------------------------------------------------------------*/

// !Call home

if ( ! function_exists( 'sf_updater_remote_plugin_infos' ) ) :
function sf_updater_remote_plugin_infos( $request ) {
	return wp_remote_post(
		'https://www.screenfeed.fr/downloads/',
		array( 'timeout' => 30, 'body' => array( 'action' => 'plugin_information', 'request' => serialize( (array) $request ) ) )
	);
}
endif;


// !Deal with sf_remote_plugin_infos() result

if ( ! function_exists( 'sf_updater_remote_retrieve_body' ) ) :
function sf_updater_remote_retrieve_body( $response ) {
	$error_msg = sprintf(
		__( 'An unexpected error occurred. Something may be wrong with screenfeed.fr or this server&#8217;s configuration. If you continue to have problems, please leave a message on <a href="%s">my blog</a>.', 'noop' ),
		'https://www.screenfeed.fr/blog/'
	);

	if ( is_wp_error( $response ) ) {
		return new WP_Error(
			'plugins_api_failed',
			$error_msg,
			$response->get_error_message()
		);
	}

	$response = wp_unslash( wp_remote_retrieve_body( $response ) );

	if ( is_serialized( $response ) ) {
		return (object) @unserialize( $response );
	}

	return new WP_Error( 'plugins_api_failed', $error_msg, $response );
}
endif;


// For WP < 3.6
if ( ! function_exists( 'wp_unslash' ) ) :
function wp_unslash( $value ) {
	return stripslashes_deep( $value );
}
endif;

/**/
