<?php
/*
 * Plugin Name: SF After Content Box
 * Plugin URI: http://www.screenfeed.fr/plugin-wp/after-content-box/
 * Description: Add a box at the end of the content of posts with a fullwidth layout. That box will contain everything other plugins may add.
 * Version: 1.0
 * Author: Grégory Viguier
 * Author URI: http://www.screenfeed.fr/
 * License: GPLv3
 * License URI: http://www.screenfeed.fr/gpl-v3.txt
 */

defined( 'ABSPATH' ) or die( 'Cheatin\' uh?' );


/*------------------------------------------------------------------------------------------------*/
/* CONSTANTS ==================================================================================== */
/*------------------------------------------------------------------------------------------------*/

define( 'SFACB_VERSION', '1.0' );


/*------------------------------------------------------------------------------------------------*/
/* ADD THE BOX ================================================================================== */
/*------------------------------------------------------------------------------------------------*/

/**
 * Filter the content very early (hopefully before extra content) and add a placeholder at the end of it.
 *
 * @version 1.0
 *
 * @param (string) $content The post content.
 *
 * @return (string) The post content.
 */
add_filter( 'the_content', 'sfacb_add_placeholder_after_content', PHP_INT_MIN );

function sfacb_add_placeholder_after_content( $content ) {
	return $content . '<div class="after-post-content-placeholder"></div>';
}


/**
 * Filter the content very late (hopefully after extra content), remove the placeholder, and maybe add the wrapper.
 *
 * @version 1.0
 *
 * @param (string) $content The post content.
 *
 * @return (string) The post content.
 */
add_filter( 'the_content', 'sfacb_add_wrapper_after_content', PHP_INT_MAX );

function sfacb_add_wrapper_after_content( $content ) {
	$placeholder = '<div class="after-post-content-placeholder"></div>';
	$post_id     = get_the_id();

	// Test if the current post is the main content.
	if ( ! sfacb_is_single_page( $post_id ) ) {
		return str_replace( $placeholder, '', $content );
	}

	// Test if the layout is a fullwidth.
	$is_fullwidth_layout = 'et_full_width_page' === get_post_meta( $post_id, '_et_pb_page_layout', true ) && 'on' === get_post_meta( $post_id, '_et_pb_use_builder', true );
	/**
	 * Filter the result of the layout test.
	 *
	 * @since 1.0
	 *
	 * @param (bool) $is_fullwidth_layout Result of the layout test.
	 * @param (int)  $post_id             The post ID.
	 */
	$is_fullwidth_layout = apply_filters( 'sfacb.content.is_fullwidth_layout', $is_fullwidth_layout, $post_id );

	if ( ! $is_fullwidth_layout ) {
		return str_replace( $placeholder, '', $content );
	}

	// Test the content.
	$trimed_content = trim( $content );

	/**
	 * 50 is the placeholder length.
	 * If the placeholder is not at the end of the content that means some plugins added something.
	 * Let's catch this new content.
	 */
	if ( substr( $trimed_content, -50 ) !== $placeholder && preg_match( "@$placeholder(.+)$@s", $content, $match ) ) {

		$wrapper = '<div class="after-post-content-outer et_pb_section"><div class="after-post-content-inner et_pb_row">%s</div></div>';
		/**
		 * Filter the wrapper html tags.
		 *
		 * @since 1.0
		 *
		 * @param (string) $wrapper The wrapper.
		 * @param (int)    $post_id The post ID.
		 */
		$wrapper = apply_filters( 'sfacb.content.wrapper_html', $wrapper, $post_id );

		/**
		 * $match[0] is the placeholder + new content.
		 * $match[1] is the new content.
		 *
		 * 1: wrap new content.
		 */
		$new_content = sprintf( $wrapper, trim( $match[1] ) );
		/**
		 * 2: replace "placeholder + new content" by "wrapped new content".
		 */
		return str_replace( $match[0], $new_content, $content );
	}

	return str_replace( $placeholder, '', $content );
}


/*------------------------------------------------------------------------------------------------*/
/* TOOLS ======================================================================================== */
/*------------------------------------------------------------------------------------------------*/

/**
 * Tell if we're displaying the main content of the page.
 * Kind of `( is_single( $post ) || is_page( $post ) || is_attachment( $post ) )` with extra tests.
 *
 * @since 1.0
 *
 * @param (int|object) $post A post ID or WP_Post object.
 *
 * @return (bool).
 */
function sfacb_is_single_page() {
	global $wp_query;

	if ( ! isset( $wp_query ) ) {
		_doing_it_wrong( __FUNCTION__, __( 'Conditional query tags do not work before the query is run. Before then, they always return false.' ), '3.1' );
		return false;
	}

	// First, are we in a single?
	if ( ! $wp_query->is_single && ! $wp_query->is_page && ! $wp_query->is_attachment ) {
		return false;
	}

	// If something is passed as parameter and is empty, return false instead of falling back to the current post.
	$args = func_get_args();

	if ( array_key_exists( 0, $args ) ) {
		if ( empty( $args[0] ) ) {
			return false;
		}

		$post = $args[0];
	}

	// If not in the loop, in the main query, or in a sidebar, return false.
	if ( ! in_the_loop() || ! $wp_query->is_main_query() || sfacb_in_dynamic_sidebar() ) {
		return false;
	}

	// Get the queried object.
	$queried_obj = $wp_query->get_queried_object();

	if ( ! sfacb_is_wp_post( $queried_obj ) || empty( $queried_obj->ID ) ) {
		return false;
	}

	// Get the post.
	if ( empty( $post ) ) {
		$post = get_post();
	}
	elseif ( is_numeric( $post ) ) {
		$post = get_post( (int) $post );
	}

	if ( ! sfacb_is_wp_post( $post ) || empty( $post->ID ) ) {
		return false;
	}

	// Finally, compare post IDs.
	return $post->ID === $queried_obj->ID;
}


/**
 * Tell if the given object is a WP_Post object.
 *
 * @since 1.0
 *
 * @param (mixed) $post What to test.
 *
 * @return (bool).
 */
function sfacb_is_wp_post( $post ) {
	return $post && is_object( $post ) && is_a( $post, 'WP_Post' );
}


/**
 * Tell if we're inside a dynamic sidebar.
 *
 * @since 1.0
 *
 * @return (bool).
 */
function sfacb_in_dynamic_sidebar() {
	global $wp_actions;

	$before = ! empty( $wp_actions['dynamic_sidebar_before'] ) ? (int) $wp_actions['dynamic_sidebar_before'] : 0;
	$after  = ! empty( $wp_actions['dynamic_sidebar_after'] )  ? (int) $wp_actions['dynamic_sidebar_after']  : 0;

	return $before !== $after;
}
