<?php
if ( ! defined( 'ABSPATH' ) ) {
	die( 'Cheatin\' uh?' );
}

/**
 * Singleton class.
 *
 * @package Move Login
 */
class SFML_Options extends SFML_Singleton {

	const VERSION      = '1.2';
	const OPTION_NAME  = 'sfml';
	const OPTION_GROUP = 'sfml_settings';
	const OPTION_PAGE  = 'move-login';

	/**
	 * Options.
	 *
	 * @var (array)
	 */
	protected $options;

	/**
	 * Default options.
	 *
	 * @var (array)
	 */
	protected $options_default;

	/**
	 * Slugs.
	 *
	 * @var (array)
	 */
	protected $slugs;

	/**
	 * Setting labels.
	 *
	 * @var (array)
	 */
	protected $labels;


	/**
	 * Init.
	 */
	public function _init() {
		if ( defined( 'WP_UNINSTALL_PLUGIN' ) ) {
			return;
		}

		// Register option and sanitization method.
		$this->register_setting();
	}


	/**
	 * Get default options.
	 *
	 * @return (array)
	 */
	public function get_default_options() {
		$this->maybe_clear_options_cache();

		if ( isset( $this->options_default ) ) {
			return $this->options_default;
		}

		// Default slugs.
		$this->options_default = array(
			'slugs.postpass'     => 'postpass',
			'slugs.logout'       => 'logout',
			'slugs.lostpassword' => 'lostpassword',
			'slugs.resetpass'    => 'resetpass',
			'slugs.register'     => 'register',
			'slugs.login'        => 'login',
		);

		// Plugins can add their own actions.
		$additional_slugs = static::get_additional_labels();

		if ( $additional_slugs && is_array( $additional_slugs ) ) {
			foreach ( $additional_slugs as $slug_key => $slug_label ) {
				$slug_key = sanitize_title( $slug_key, '', 'display' );

				if ( ! empty( $slug_key ) && ! isset( $this->options_default[ 'slug.' . $slug_key ] ) ) {
					$this->options_default[ 'slugs.' . $slug_key ] = $slug_key;
				}
			}
		}

		// Options.
		$this->options_default = array_merge( $this->options_default, array(
			'deny_wp_login_access' => 1,
			'deny_admin_access'    => 0,
		) );

		return $this->options_default;
	}


	/**
	 * Get all options.
	 *
	 * @return (array)
	 */
	public function get_options() {
		$this->maybe_clear_options_cache();

		if ( isset( $this->options ) ) {
			return $this->options;
		}

		$this->options = array();
		$old_options   = get_site_option( static::OPTION_NAME );
		$defaults      = $this->get_default_options();

		if ( is_array( $old_options ) ) {
			$default_slugs = static::get_sub_options( 'slugs', $defaults );

			// Add and escape slugs.
			foreach ( $default_slugs as $slug_key => $default_slug ) {
				$this->options[ 'slugs.' . $slug_key ] = ! empty( $old_options[ 'slugs.' . $slug_key ] ) ? sanitize_title( $old_options[ 'slugs.' . $slug_key ], $default_slug, 'display' ) : $default_slug;
			}

			// Add and escape other options.
			if ( isset( $defaults['deny_wp_login_access'] ) ) {
				$this->options['deny_wp_login_access'] = isset( $old_options['deny_wp_login_access'] ) ? min( 3, max( 1, (int) $old_options['deny_wp_login_access'] ) ) : $defaults['deny_wp_login_access'];
			}

			if ( isset( $defaults['deny_admin_access'] ) ) {
				$this->options['deny_admin_access'] = isset( $old_options['deny_admin_access'] ) ? min( 3, max( 0, (int) $old_options['deny_admin_access'] ) ) : $defaults['deny_admin_access'];
			}
		} else {
			$this->options = $defaults;
		}

		// Generic filter, change the values.
		$options_tmp   = apply_filters( 'sfml_options', $this->options );
		// Make sure no keys have been added or removed.
		$this->options = array_intersect_key( array_merge( $this->options, $options_tmp ), $this->options );

		return $this->options;
	}


	/**
	 * Get an option.
	 *
	 * @since 2.4
	 *
	 * @param (string) $option_name Name of the option.
	 *
	 * @return (mixed) Return null if the option dosn't exist.
	 */
	public function get_option( $option_name ) {
		$options = $this->get_options();
		return $option_name && isset( $options[ $option_name ] ) ? $options[ $option_name ] : null;
	}


	/**
	 * Get the slugs.
	 *
	 * @return (array)
	 */
	public function get_slugs() {
		$this->maybe_clear_options_cache();

		if ( ! isset( $this->slugs ) ) {
			$this->slugs = static::get_sub_options( 'slugs', $this->get_options() );
		}

		return $this->slugs;
	}


	/**
	 * Setting field labels for the slugs.
	 *
	 * @return (array)
	 */
	public function get_slug_field_labels() {
		$this->maybe_clear_options_cache();

		if ( isset( $this->labels ) ) {
			return $this->labels;
		}

		$this->labels = array(
			'login'        => __( 'Log in' ),
			'logout'       => __( 'Log out' ),
			'register'     => __( 'Register' ),
			'lostpassword' => __( 'Lost Password' ),
			'resetpass'    => __( 'Password Reset' ),
		);

		$new_actions = static::get_additional_labels();

		if ( $new_actions ) {
			$new_actions  = array_diff_key( $new_actions, $this->labels );
			$this->labels = array_merge( $this->labels, $new_actions );
		}

		return $this->labels;
	}


	/**
	 * Return the "other" original login actions: not the ones listed in our settings.
	 *
	 * @return (array)
	 */
	public function get_other_actions() {
		return array_diff_key( array(
			'retrievepassword' => 'retrievepassword',
			'rp'               => 'rp',
		), $this->get_slug_field_labels() );
	}


	/**
	 * Clear options cache.
	 *
	 * @param (bool) $force Clear the cache manually.
	 */
	public function maybe_clear_options_cache( $force = false ) {
		$clear = false;
		/**
		 * Clear options cache.
		 *
		 * @param (bool) $clear Return true if you want to clear the cache.
		 */
		if ( $force || apply_filters( static::OPTION_NAME . '_clear_options_cache', $clear ) ) {
			$this->options         = null;
			$this->options_default = null;
			$this->slugs           = null;
			$this->labels          = null;
			remove_all_filters( static::OPTION_NAME . '_clear_options_cache' );
		}
	}


	/**
	 * An improved version of `register_setting()`, that always exists and that works for network options.
	 */
	protected function register_setting() {
		global $new_whitelist_options;

		$sanitize_callback = array( $this, 'sanitize_options' );

		if ( ! is_multisite() ) {
			if ( function_exists( 'register_setting' ) ) {
				register_setting( static::OPTION_GROUP, static::OPTION_NAME, $sanitize_callback );
				return;
			}

			$new_whitelist_options = isset( $new_whitelist_options ) && is_array( $new_whitelist_options ) ? $new_whitelist_options : array(); // WPCS: override ok.
			$new_whitelist_options[ static::OPTION_GROUP ]   = isset( $new_whitelist_options[ static::OPTION_GROUP ] ) && is_array( $new_whitelist_options[ static::OPTION_GROUP ] ) ? $new_whitelist_options[ static::OPTION_GROUP ] : array();
			$new_whitelist_options[ static::OPTION_GROUP ][] = static::OPTION_NAME;
		} elseif ( is_admin() ) {
			$whitelist = sfml_cache_data( 'new_whitelist_network_options' );
			$whitelist = is_array( $whitelist ) ? $whitelist : array();
			$whitelist[ static::OPTION_GROUP ]   = isset( $whitelist[ static::OPTION_GROUP ] ) ? $whitelist[ static::OPTION_GROUP ] : array();
			$whitelist[ static::OPTION_GROUP ][] = static::OPTION_NAME;
			sfml_cache_data( 'new_whitelist_network_options', $whitelist );
		}

		if ( $sanitize_callback ) {
			add_filter( 'sanitize_option_' . static::OPTION_NAME, $sanitize_callback );
		}
	}


	/**
	 * Sanitize options on save.
	 *
	 * @param (array) $options Options to sanitize.
	 *
	 * @return (array)
	 */
	public function sanitize_options( $options = array() ) {
		$errors            = array( 'forbidden' => array(), 'duplicates' => array() );
		$old_options       = get_site_option( static::OPTION_NAME );
		$default_options   = $this->get_default_options();
		$sanitized_options = array();

		// Add and sanitize slugs.
		$default_slugs = static::get_sub_options( 'slugs', $default_options );
		$exclude       = $this->get_other_actions();

		foreach ( $default_slugs as $slug_key => $default_slug ) {

			if ( isset( $exclude[ $slug_key ] ) ) {
				$sanitized_options[ 'slugs.' . $slug_key ] = $exclude[ $slug_key ];
				continue;
			}

			$sanitized_options[ 'slugs.' . $slug_key ] = false;

			if ( ! empty( $options[ 'slugs.' . $slug_key ] ) ) {
				$tmp_slug = sanitize_title( $options[ 'slugs.' . $slug_key ], $default_slug );

				// 'postpass', 'retrievepassword' and 'rp' are forbidden.
				if ( in_array( $tmp_slug, $exclude, true ) ) {
					$errors['forbidden'][] = $tmp_slug;
				}
				// Make sure the slug is not already set for another action.
				elseif ( in_array( $tmp_slug, $sanitized_options, true ) ) {
					$errors['duplicates'][] = $tmp_slug;
				}
				// Yay!
				else {
					$sanitized_options[ 'slugs.' . $slug_key ] = $tmp_slug;
				}
			}

			// Fallback to old value or default value.
			if ( ! $sanitized_options[ 'slugs.' . $slug_key ] ) {
				if ( ! isset( $exclude[ $slug_key ] ) && ! empty( $old_options[ 'slugs.' . $slug_key ] ) ) {
					$sanitized_options[ 'slugs.' . $slug_key ] = sanitize_title( $old_options[ 'slugs.' . $slug_key ], $default_slug );
				} else {
					$sanitized_options[ 'slugs.' . $slug_key ] = $default_slug;
				}
			}
		}

		// Add and sanitize other options.
		if ( isset( $default_options['deny_wp_login_access'] ) ) {
			if ( isset( $options['deny_wp_login_access'] ) ) {

				$sanitized_options['deny_wp_login_access'] = min( 3, max( 1, (int) $options['deny_wp_login_access'] ) );

			} elseif ( isset( $old_options['deny_wp_login_access'] ) ) {

				$sanitized_options['deny_wp_login_access'] = min( 3, max( 1, (int) $old_options['deny_wp_login_access'] ) );

			} else {
				$sanitized_options['deny_wp_login_access'] = $default_options['deny_wp_login_access'];
			}
		}

		if ( isset( $default_options['deny_admin_access'] ) ) {
			if ( isset( $options['deny_admin_access'] ) ) {

				$sanitized_options['deny_admin_access'] = min( 3, max( 0, (int) $options['deny_admin_access'] ) );

			} elseif ( isset( $old_options['deny_admin_access'] ) ) {

				$sanitized_options['deny_admin_access'] = min( 3, max( 0, (int) $old_options['deny_admin_access'] ) );

			} else {
				$sanitized_options['deny_admin_access'] = $default_options['deny_admin_access'];
			}
		}

		/**
		 * Filter the options after being sanitized.
		 *
		 * @param (array) $sanitized_options The new options, sanitized.
		 * @param (array) $options           The submitted options.
		 */
		$options_tmp       = apply_filters( 'sfml_sanitize_options', $sanitized_options, $options );
		// Make sure no keys have been removed.
		$sanitized_options = array_merge( $sanitized_options, $options_tmp );

		// Clear options cache.
		$this->maybe_clear_options_cache( true );

		// Add the rewrite rules to the `.htaccess`/`web.config` file.
		$old_slugs = static::get_sub_options( 'slugs', $old_options );
		$new_slugs = static::get_sub_options( 'slugs', $sanitized_options );

		if ( $old_slugs !== $new_slugs ) {
			sfml_include_rewrite_file();
			sfml_write_rules( sfml_rules( $new_slugs ) );
		}

		// Trigger errors.
		if ( is_admin() ) {
			$errors['forbidden']  = array_unique( $errors['forbidden'] );
			$errors['duplicates'] = array_unique( $errors['duplicates'] );

			if ( $nbr_forbidden = count( $errors['forbidden'] ) ) {
				/** Translators: %s is an URL slug name. */
				add_settings_error( 'sfml_settings', 'forbidden-slugs', sprintf( _n( 'The slug %s is forbidden.', 'The slugs %s are forbidden.', $nbr_forbidden, 'sf-move-login' ), wp_sprintf( '<code>%l</code>', $errors['forbidden'] ) ) );
			}
			if ( ! empty( $errors['duplicates'] ) ) {
				add_settings_error( 'sfml_settings', 'duplicates-slugs', __( 'The links can\'t have the same slugs.', 'sf-move-login' ) );
			}
		}

		return $sanitized_options;
	}


	/**
	 * Get sub-options.
	 *
	 * For example:
	 * static::get_sub_options( 'foo', array(
	 *     'option1'     => 'value1',
	 *     'foo.option2' => 'value2',
	 *     'foo.option3' => 'value3',
	 * ) );
	 * Will return:
	 * array(
	 *     'option2' => 'value2',
	 *     'option3' => 'value3',
	 * )
	 *
	 * @param (string) $name    The sub-option name.
	 * @param (array)  $options Array of options.
	 *
	 * @return (array)
	 */
	public static function get_sub_options( $name, $options ) {
		if ( ! $options || ! $name ) {
			return array();
		}

		$options = (array) $options;

		if ( isset( $options[ $name ] ) ) {
			return $options[ $name ];
		}

		$group = array();
		$name  = rtrim( $name, '.' ) . '.';

		foreach ( $options as $k => $v ) {
			if ( 0 === strpos( $k, $name ) ) {
				$group[ substr( $k, strlen( $name ) ) ] = $v;
			}
		}

		return ! empty( $group ) ? $group : null;
	}


	/**
	 * Get custom labels (added by other plugins).
	 *
	 * @since 2.4
	 *
	 * @return (array)
	 */
	public static function get_additional_labels() {
		$new_actions = array();
		/**
		 * Plugins can add their own actions.
		 *
		 * @param (array) $new_actions Custom actions.
		 */
		return apply_filters( 'sfml_additional_slugs', $new_actions );
	}
}
