<?php
if ( ! defined( 'ABSPATH' ) ) {
	die( 'Cheatin\' uh?' );
}

if ( ! function_exists( 'set_url_scheme' ) ) :
	/**
	 * Sets the scheme for a URL.
	 *
	 * @since WP 3.4.0
	 * @since WP 4.4.0 The 'rest' scheme was added.
	 *
	 * @param (string)      $url    Absolute URL that includes a scheme.
	 * @param (string|null) $scheme Optional. Scheme to give $url. Currently 'http', 'https', 'login',
	 *                              'login_post', 'admin', 'relative', 'rest', 'rpc', or null. Default null.
	 * @return (string) $url URL with chosen scheme.
	 */
	function set_url_scheme( $url, $scheme = null ) {
		$orig_scheme = $scheme;

		if ( ! $scheme ) {
			$scheme = is_ssl() ? 'https' : 'http';
		} elseif ( 'admin' === $scheme || 'login' === $scheme || 'login_post' === $scheme || 'rpc' === $scheme ) {
			$scheme = is_ssl() || force_ssl_admin() ? 'https' : 'http';
		} elseif ( 'http' !== $scheme && 'https' !== $scheme && 'relative' !== $scheme ) {
			$scheme = is_ssl() ? 'https' : 'http';
		}

		$url = trim( $url );

		if ( substr( $url, 0, 2 ) === '//' ) {
			$url = 'http:' . $url;
		}

		if ( 'relative' === $scheme ) {
			$url = ltrim( preg_replace( '#^\w+://[^/]*#', '', $url ) );

			if ( '' !== $url && '/' === $url[0] ) {
				$url = '/' . ltrim( $url , "/ \t\n\r\0\x0B" );
			}
		} else {
			$url = preg_replace( '#^\w+://#', $scheme . '://', $url );
		}

		/**
		 * Filters the resulting URL after setting the scheme.
		 *
		 * @since WP 3.4.0
		 *
		 * @param (string)      $url         The complete URL including scheme and path.
		 * @param (string)      $scheme      Scheme applied to the URL. One of 'http', 'https', or 'relative'.
		 * @param (string|null) $orig_scheme Scheme requested for the URL. One of 'http', 'https', 'login',
		 *                                   'login_post', 'admin', 'relative', 'rest', 'rpc', or null.
		 */
		return apply_filters( 'set_url_scheme', $url, $scheme, $orig_scheme );
	}
endif;


if ( ! function_exists( 'wp_is_writable' ) ) :
	/**
	 * Determine if a directory is writable.
	 *
	 * This function is used to work around certain ACL issues in PHP primarily affecting Windows Servers.
	 *
	 * @since WP 3.6.0
	 * @see win_is_writable()
	 *
	 * @param (string) $path Path to check for write-ability.
	 *
	 * @return (bool) Whether the path is writable.
	 */
	function wp_is_writable( $path ) {
		if ( 'WIN' === strtoupper( substr( PHP_OS, 0, 3 ) ) ) {
			return win_is_writable( $path );
		}
		return @is_writable( $path );
	}
endif;
